<?php
// controllers/ForgotPasswordController.php

require_once dirname(__DIR__) . '/controllers/BaseController.php';
require_once dirname(__DIR__) . '/models/Users.php';

class ForgotPasswordController extends BaseController {

    public function __construct(mysqli $conn) {
        parent::__construct($conn);
    }

    /**
     * Menampilkan formulir permintaan reset password (masukkan email).
     */
    public function requestForm() {
        // Jika user sudah login, arahkan ke home
        if (isset($_SESSION['user_id'])) {
            $this->redirect('home');
        }
        $this->view('forgot_password_request');
    }

    /**
     * Memproses permintaan reset password.
     * Mengirim link reset (simulasi email).
     */
    public function sendResetLink() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $email = $_POST['email'] ?? '';

            if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $this->setErrorMessage("Email tidak valid.");
                $this->redirect('forgot_password_request');
                exit();
            }

            $userModel = new Users($this->conn);
            $user = $userModel->getUserByEmail($email);

            if ($user) {
                // Buat token unik
                $token = bin2hex(random_bytes(32)); // Token 64 karakter heksadesimal

                // Simpan token di database
                if ($userModel->createPasswordResetToken($email, $token)) {
                    // --- SIMULASI PENGIRIMAN EMAIL ---
                    // KITA SEKARANG AKAN MENYIMPAN HANYA TOKENNYA DI SESI
                    // Halaman forgot_password_sent.php akan langsung menggunakan token ini
                    // untuk redirect ke form reset.
                    $_SESSION['reset_token_for_auto_redirect'] = $token;
                    // --- AKHIR SIMULASI ---

                    $this->setSuccessMessage("Link reset password telah diproses. Silakan lanjutkan untuk mengatur ulang sandi Anda.");
                    $this->redirect('forgot_password_sent'); // Halaman konfirmasi / redirect otomatis
                } else {
                    // Jika pembuatan token gagal di database
                    error_log("Failed to create password reset token for email: " . $email); // Log error
                    $this->setErrorMessage("Gagal membuat token reset. Silakan coba lagi.");
                    $this->redirect('forgot_password_request');
                }
            } else {
                // Untuk keamanan, selalu berikan pesan yang generik agar tidak membocorkan keberadaan email.
                $this->setInfoMessage("Jika email Anda terdaftar di sistem kami, silakan lanjutkan untuk mengatur ulang sandi Anda.");
                // Jika email tidak ditemukan, kita tidak punya token untuk redirect otomatis.
                unset($_SESSION['reset_token_for_auto_redirect']); 

                $this->redirect('forgot_password_sent'); 
            }
        } else {
            $this->redirect('forgot_password_request');
        }
    }

    /**
     * Menampilkan halaman konfirmasi dan secara otomatis mengarahkan ke form reset.
     * Ini bertindak sebagai perantara "pengiriman email" virtual.
     */
    public function sentConfirmation() {
        // Cek apakah ada token yang tersimpan di sesi untuk auto-redirect
        $token_to_redirect = $_SESSION['reset_token_for_auto_redirect'] ?? null;
        
        // Hapus token dari sesi setelah diambil agar tidak bisa digunakan lagi secara otomatis
        // atau menyebabkan loop redirect.
        unset($_SESSION['reset_token_for_auto_redirect']); 

        if ($token_to_redirect) {
            // Jika ada token, langsung redirect ke form reset password
            $this->redirect('forgot_password_reset_form', ['token' => $token_to_redirect]);
        } else {
            // Jika tidak ada token (misal karena email tidak ditemukan),
            // tampilkan pesan info saja dan tombol kembali ke login.
            $this->view('forgot_password_sent', ['show_manual_button' => true]);
        }
    }

    /**
     * Menampilkan formulir untuk mengatur password baru setelah mengklik link reset.
     *
     * @param string|null $token Token reset password dari URL.
     */
    public function resetForm($token = null) {
        if ($token === null) {
            $this->setErrorMessage("Token reset password tidak ditemukan.");
            $this->redirect('login');
            exit();
        }

        $userModel = new Users($this->conn);
        $email = $userModel->verifyPasswordResetToken($token);

        if (!$email) {
            $this->setErrorMessage("Token reset password tidak valid atau sudah kadaluarsa.");
            $this->redirect('login');
            exit();
        }

        // Simpan token dan email di sesi sementara agar bisa digunakan saat submit form
        $_SESSION['reset_token'] = $token;
        $_SESSION['reset_email'] = $email;

        $this->view('forgot_password_reset', ['token' => $token, 'email' => $email]);
    }

    /**
     * Memproses pengiriman password baru.
     * Memverifikasi token dan mengupdate password di database.
     */
    public function resetPassword() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $token = $_POST['token'] ?? '';
            $email = $_POST['email'] ?? '';
            $new_password = $_POST['new_password'] ?? '';
            $confirm_password = $_POST['confirm_password'] ?? '';

            // Pastikan token dan email ada di sesi (dari resetForm) untuk keamanan ekstra
            if (!isset($_SESSION['reset_token']) || $_SESSION['reset_token'] !== $token ||
                !isset($_SESSION['reset_email']) || $_SESSION['reset_email'] !== $email) {
                $this->setErrorMessage("Sesi reset password tidak valid atau sudah berakhir. Silakan ulangi prosesnya.");
                $this->redirect('login');
                exit();
            }

            if (empty($new_password) || $new_password !== $confirm_password || strlen($new_password) < 6) {
                $this->setErrorMessage("Password baru tidak cocok atau terlalu pendek (minimal 6 karakter).");
                $this->redirect('forgot_password_reset_form', ['token' => $token]);
                exit();
            }

            $userModel = new Users($this->conn);
            
            // Verifikasi token lagi untuk memastikan masih valid saat submit
            $verified_email = $userModel->verifyPasswordResetToken($token);

            if ($verified_email && $verified_email === $email) {
                // Update password
                if ($userModel->updatePasswordByEmail($email, $new_password)) {
                    // Hapus token setelah berhasil digunakan
                    $userModel->deletePasswordResetToken($token);
                    // Hapus data sesi reset
                    unset($_SESSION['reset_token']);
                    unset($_SESSION['reset_email']);

                    $this->setSuccessMessage("Password Anda berhasil direset. Silakan login dengan password baru.");
                    $this->redirect('login');
                } else {
                    $this->setErrorMessage("Gagal mengupdate password. Silakan coba lagi.");
                    $this->redirect('forgot_password_reset_form', ['token' => $token]);
                }
            } else {
                $this->setErrorMessage("Token reset password tidak valid atau sudah kadaluarsa. Silakan ulangi proses lupa sandi.");
                $this->redirect('forgot_password_request');
            }
        } else {
            $this->redirect('forgot_password_request');
        }
    }
}
