<?php
// controllers/KategoriController.php

// Pastikan semua model dan BaseController di-include dengan jalur yang benar
require_once dirname(__DIR__) . '/models/Kategori.php'; // Model Kategori
require_once dirname(__DIR__) . '/controllers/BaseController.php'; // BaseController
// Diperlukan untuk pengecekan destroy method (jika ada film terkait)
require_once dirname(__DIR__) . '/models/Film.php'; 

// Pastikan kelas ini mewarisi dari BaseController
class KategoriController extends BaseController {

    /**
     * Konstruktor untuk KategoriController.
     * Menerima objek koneksi MySQLi dan meneruskannya ke BaseController.
     * @param mysqli $conn Objek koneksi database MySQLi.
     */
    public function __construct(mysqli $conn) {
        parent::__construct($conn);
    }

    /**
     * Menampilkan daftar semua kategori.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kategori
     */
    public function index() {
        // Memerlukan peran admin
        $this->requireAdmin();

        $kategoriModel = new Kategori($this->conn); // Meneruskan koneksi
        $kategoris = $kategoriModel->getAllKategoris();

        // Load view untuk menampilkan daftar kategori
        $this->view('kategori', ['kategoris' => $kategoris]);
    }

    /**
     * Menampilkan formulir untuk menambah kategori baru.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kategori_create
     */
    public function create() {
        $this->requireAdmin();

        $this->view('kategori_form'); // views/kategori_form.php untuk tambah kategori
    }

    /**
     * Memproses data POST dari formulir penambahan kategori baru.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kategori_store
     */
    public function store() {
        $this->requireAdmin();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $kategori_nama = $_POST['kategori'] ?? ''; // Menggunakan nama variabel yang lebih spesifik

            if (empty($kategori_nama)) {
                $this->setErrorMessage("Nama kategori harus diisi.");
                $this->redirect('kategori_create');
                exit();
            }

            $kategoriModel = new Kategori($this->conn); // Meneruskan koneksi
            $result = $kategoriModel->addKategori($kategori_nama);

            if ($result) {
                $this->setSuccessMessage("Kategori berhasil ditambahkan.");
                $this->redirect('kategori'); // Redirect ke daftar kategori
                exit();
            } else {
                $this->setErrorMessage("Gagal menambahkan kategori. Nama kategori mungkin sudah ada.");
                $this->redirect('kategori_create'); // Kembali ke form
                exit();
            }
        } else {
            $this->redirect('kategori_create');
            exit();
        }
    }

    /**
     * Menampilkan formulir untuk mengedit kategori yang sudah ada.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kategori_edit&id=X
     */
    public function edit($kategori_id) {
        $this->requireAdmin();

        $kategoriModel = new Kategori($this->conn); // Meneruskan koneksi
        $kategori = $kategoriModel->getKategoriById($kategori_id);

        if (!$kategori) {
            $this->setErrorMessage("Kategori tidak ditemukan.");
            $this->redirect('kategori');
            exit();
        }

        $this->view('kategori_form_update', ['kategori' => $kategori]);
    }

    /**
     * Memproses data POST dari formulir pembaruan kategori.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kategori_update
     */
    public function update() {
        $this->requireAdmin();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $id = $_POST['id'] ?? null;
            $kategori_nama = $_POST['kategori'] ?? '';

            if (empty($id) || empty($kategori_nama)) {
                $this->setErrorMessage("Data tidak lengkap untuk pembaruan kategori.");
                $this->redirect('kategori_edit', ['id' => $id]);
                exit();
            }

            $kategoriModel = new Kategori($this->conn); // Meneruskan koneksi
            $result = $kategoriModel->updateKategori($id, $kategori_nama);

            if ($result) {
                $this->setSuccessMessage("Kategori berhasil diperbarui.");
                $this->redirect('kategori');
                exit();
            } else {
                $this->setErrorMessage("Gagal memperbarui kategori. Nama kategori mungkin sudah ada.");
                $this->redirect('kategori_edit', ['id' => $id]);
                exit();
            }
        } else {
            $this->redirect('kategori');
            exit();
        }
    }

    /**
     * Menghapus kategori dari database.
     * Hanya untuk admin.
     * Dipanggil oleh index.php?hal=kategori_destroy&id=X
     */
    public function destroy($kategori_id) {
        $this->requireAdmin();

        $kategoriModel = new Kategori($this->conn); // Meneruskan koneksi
        $filmModel = new Film($this->conn); // Diperlukan untuk mengecek film terkait

        // Cek apakah ada film yang terkait dengan kategori ini
        if (count($filmModel->getFilmsByKategoriId($kategori_id)) > 0) { // Asumsi ada method ini di FilmModel
            $this->setErrorMessage("Tidak bisa menghapus kategori. Ada film terkait kategori ini.");
            $this->redirect('kategori');
            exit();
        }

        $result = $kategoriModel->deleteKategori($kategori_id);

        if ($result) {
            $this->setSuccessMessage("Kategori berhasil dihapus.");
        } else {
            $this->setErrorMessage("Gagal menghapus kategori. Kendala database.");
        }

        $this->redirect('kategori');
        exit();
    }
}
