<?php
// models/Kategori.php

// Pastikan BaseModel.php di-include. Sesuaikan path jika struktur direktori Anda berbeda.
require_once dirname(__DIR__) . '/models/BaseModel.php';

class Kategori extends BaseModel {
    // Properti yang mendefinisikan nama tabel di database
    protected $table = 'kategori';

    /**
     * Mengambil semua data kategori dari database.
     *
     * @return array Array asosiatif berisi semua data kategori.
     */
    public function getAllKategoris() {
        $sql = "SELECT * FROM {$this->table} ORDER BY kategori ASC"; // Urutkan berdasarkan nama kategori
        $result = $this->conn->query($sql);
        $kategoris = [];
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $kategoris[] = $row;
            }
        }
        return $kategoris;
    }

    /**
     * Mengambil data kategori berdasarkan ID.
     *
     * @param int $id ID kategori.
     * @return array|null Array asosiatif data kategori jika ditemukan, null jika tidak.
     */
    public function getKategoriById($id) {
        $stmt = $this->conn->prepare("SELECT * FROM {$this->table} WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }

    /**
     * Menambahkan kategori baru ke database.
     *
     * @param string $kategori Nama kategori.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function addKategori($kategori) {
        $stmt = $this->conn->prepare("INSERT INTO {$this->table} (kategori) VALUES (?)");
        $stmt->bind_param("s", $kategori);
        return $stmt->execute();
    }

    /**
     * Memperbarui data kategori yang sudah ada di database.
     *
     * @param int $id ID kategori yang akan diperbarui.
     * @param string $kategori Nama kategori baru.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function updateKategori($id, $kategori) {
        $stmt = $this->conn->prepare("UPDATE {$this->table} SET kategori = ? WHERE id = ?");
        $stmt->bind_param("si", $kategori, $id);
        return $stmt->execute();
    }

    /**
     * Menghapus kategori dari database berdasarkan ID.
     * Perhatikan FOREIGN KEY CONSTRAINTS. Jika ada film terkait kategori ini,
     * penghapusan mungkin akan diblokir.
     *
     * @param int $id ID kategori yang akan dihapus.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function deleteKategori($id) {
        $stmt = $this->conn->prepare("DELETE FROM {$this->table} WHERE id = ?");
        $stmt->bind_param("i", $id);
        return $stmt->execute();
    }

    /**
     * Menghitung total jumlah baris di tabel kategori.
     * @return int Jumlah total kategori.
     */
    public function countAll() {
        $result = $this->conn->query("SELECT COUNT(*) FROM {$this->table}");
        if (!$result) {
            error_log("SQL Error in Kategori::countAll: " . $this->conn->error);
            return 0;
        }
        $row = $result->fetch_row();
        return $row[0];
    }
}
