    <?php
    // views/film.php
    // View ini digunakan untuk menampilkan daftar semua film.
    // Variabel yang diharapkan: $films (array asosiatif berisi semua data film)
    // Variabel ini disediakan oleh FilmController->listFilms() atau FilmController->index()

    // Pesan sukses atau error dari sesi (jika ada)
    if (isset($_SESSION['error_message'])) {
        echo "<p class='message error'>" . htmlspecialchars($_SESSION['error_message']) . "</p>";
        unset($_SESSION['error_message']);
    }
    if (isset($_SESSION['success_message'])) {
        echo "<p class='message success'>" . htmlspecialchars($_SESSION['success_message']) . "</p>";
        unset($_SESSION['success_message']);
    }

    // Cek status login dan peran pengguna
    $is_admin = isset($_SESSION['user_id']) && $_SESSION['user_role'] === 'admin';
    ?>

    <div class="film-list-container">
        <h1 class="page-title">Film Tayang</h1>
        <p class="page-description">Temukan film-film terbaru dan terpopuler yang sedang tayang di bioskop kami.</p>

        <?php if ($is_admin): // Hanya tampilkan tombol "Tambah Film Baru" jika admin ?>
            <div class="action-bar">
                <a href="index.php?hal=film_create" class="add-film-button">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-plus-circle"><circle cx="12" cy="12" r="10"></circle><line x1="12" y1="8" x2="12" y2="16"></line><line x1="8" y1="12" x2="16" y2="12"></line></svg>
                    Tambah Film Baru
                </a>
            </div>
        <?php endif; ?>

        <?php if (!empty($films)): ?>
            <div class="film-cards-grid">
                <?php foreach ($films as $film): ?>
                    <div class="film-card">
                        <div class="film-card-header">
                            <?php
                            // Pastikan kunci 'cover' ada sebelum mengaksesnya
                            $cover_path = isset($film['cover']) && $film['cover'] ? "assets/img/film_covers/" . htmlspecialchars(basename($film['cover'])) : null;
                            ?>
                            <?php if ($cover_path && file_exists(dirname(__DIR__) . '/' . $cover_path)): ?>
                                <img src="<?= $cover_path ?>" alt="Cover Film" class="film-cover">
                            <?php else: ?>
                                <div class="no-cover-placeholder">No Cover</div>
                            <?php endif; ?>
                            <div class="film-info-overlay">
                                <span class="film-category"><?php echo htmlspecialchars($film['kategori'] ?? 'N/A'); ?></span>
                                <span class="film-studio"><?php echo htmlspecialchars($film['studio_nama'] ?? 'N/A'); ?></span>
                            </div>
                        </div>
                        <div class="film-card-body">
                            <h2 class="film-card-title"><?php echo htmlspecialchars($film['judul'] ?? 'Judul Tidak Diketahui'); ?></h2>
                            <p class="film-release-date">Rilis: <?php echo htmlspecialchars($film['tanggal_rilis'] ?? 'N/A'); ?></p>
                            <?php
                            // Pastikan kunci 'sinopsis' ada dan potong
                            $sinopsis_text = $film['sinopsis'] ?? '';
                            $short_sinopsis = htmlspecialchars(substr($sinopsis_text, 0, 100)) . (strlen($sinopsis_text) > 100 ? '...' : '');
                            ?>
                            <p class="film-sinopsis-short"><?php echo $short_sinopsis; ?></p>
                            <div class="film-card-actions">
                                <a href="index.php?hal=film_detail&id=<?php echo htmlspecialchars($film['id']); ?>" class="action-btn action-btn-primary">
                                    Lihat Jadwal & Pesan
                                </a>
                                <?php if ($is_admin): ?>
                                    <a href="index.php?hal=film_edit&id=<?php echo htmlspecialchars($film['id']); ?>" class="action-btn action-btn-edit">
                                        Edit
                                    </a>
                                    <form action="index.php?hal=film_destroy&id=<?php echo htmlspecialchars($film['id']); ?>" method="POST"
                                            class="delete-form" onsubmit="return confirm('Apakah Anda yakin ingin menghapus film ini? Semua jadwal dan pemesanan terkait film ini juga akan dihapus.');">
                                        <button type="submit" class="action-btn action-btn-danger">
                                            Hapus
                                        </button>
                                    </form>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div class="empty-state">
                <svg xmlns="http://www.w3.org/2000/svg" width="60" height="60" viewBox="0 0 24 24" fill="none" stroke="#ccc" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-film"><rect x="2" y="2" width="20" height="20" rx="2.18" ry="2.18"></rect><line x1="7" y1="2" x2="7" y2="22"></line><line x1="17" y1="2" x2="17" y2="22"></line><line x1="2" y1="12" x2="22" y2="12"></line><line x1="2" y1="7" x2="7" y2="7"></line><line x1="2" y1="17" x2="7" y2="17"></line><line x1="17" y1="7" x2="22" y2="7"></line><line x1="17" y1="17" x2="22" y2="17"></line></svg>
                <p>Belum ada film yang tersedia saat ini.</p>
                <?php if ($is_admin): ?>
                    <a href="index.php?hal=film_create" class="action-button">Tambah Film Pertama Anda!</a>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <style>
        /* Global/Base Styles (re-confirm from previous versions) */
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f0f2f5;
            color: #333;
            line-height: 1.6;
            margin: 0;
            padding: 0;
        }

        /* Message Styling (Success/Error) */
        .message {
            padding: 12px 20px;
            margin: 20px auto;
            border-radius: 8px;
            text-align: center;
            font-weight: 600;
            max-width: 900px;
            box-sizing: border-box;
            animation: fadeInDown 0.5s ease-out;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        @keyframes fadeInDown {
            from { opacity: 0; transform: translateY(-20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .message.error {
            background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb;
        }
        .message.success {
            background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb;
        }

        /* Main Container for the Page */
        .film-list-container {
            max-width: 1200px; /* Lebih lebar untuk grid kartu */
            margin: 30px auto;
            padding: 30px;
            background-color: #ffffff;
            border-radius: 12px;
            box-shadow: 0 6px 20px rgba(0, 0, 0, 0.1);
            animation: fadeIn 0.8s ease-out;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .page-title {
            text-align: center;
            color: #1a2a3a;
            font-size: 2.8em;
            font-weight: 800;
            margin-bottom: 10px;
            position: relative;
            padding-bottom: 15px;
        }
        .page-title::after {
            content: '';
            width: 80px;
            height: 4px;
            background-color: #007bff;
            position: absolute;
            bottom: 0;
            left: 50%;
            transform: translateX(-50%);
            border-radius: 2px;
        }

        .page-description {
            text-align: center;
            color: #6a737d;
            font-size: 1.1em;
            margin-bottom: 30px;
            line-height: 1.5;
        }

        /* Action Bar (Tambah Film Baru) */
        .action-bar {
            text-align: right;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid #eee;
        }
        .add-film-button {
            background-color: #4CAF50; /* Green (similar to default success) */
            color: white;
            padding: 10px 20px;
            border-radius: 30px;
            text-decoration: none;
            font-weight: 600;
            font-size: 0.95em;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: background-color 0.2s ease, transform 0.2s ease;
            box-shadow: 0 3px 8px rgba(76,175,80,0.2);
        }
        .add-film-button:hover {
            background-color: #45a049;
            transform: translateY(-2px);
        }
        .add-film-button svg {
            vertical-align: middle;
        }

        /* Film Cards Grid */
        .film-cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); /* Kolom adaptif */
            gap: 30px; /* Jarak antar kartu */
            justify-content: center; /* Pusatkan kartu jika kurang dari full row */
            padding: 20px 0;
        }

        .film-card {
            background-color: #ffffff;
            border-radius: 12px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            overflow: hidden; /* Pastikan gambar dan konten tidak keluar */
            display: flex;
            flex-direction: column;
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .film-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.15);
        }

        .film-card-header {
            position: relative;
            width: 100%;
            height: 375px; /* Proporsi poster standar (2:3) untuk lebar 250px */
            overflow: hidden;
            background-color: #f0f0f0; /* Fallback for no cover */
        }
        .film-cover {
            width: 100%;
            height: 100%;
            object-fit: cover; /* Pastikan gambar mengisi area tanpa distorsi */
            display: block;
        }
        .no-cover-placeholder {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.8em;
            color: #aaa;
            background-color: #e0e0e0;
        }

        .film-info-overlay {
            position: absolute;
            top: 10px;
            left: 10px;
            right: 10px;
            display: flex;
            justify-content: space-between;
            gap: 8px;
            z-index: 1; /* Pastikan di atas cover */
        }
        .film-category, .film-studio {
            background-color: rgba(0, 0, 0, 0.7); /* Background transparan gelap */
            color: white;
            padding: 5px 10px;
            border-radius: 5px;
            font-size: 0.8em;
            font-weight: 500;
            white-space: nowrap; /* Jangan putus baris */
        }
        .film-category {
            background-color: rgba(0, 123, 255, 0.8); /* Blue for category */
        }
        .film-studio {
            background-color: rgba(108, 117, 125, 0.8); /* Grey for studio */
        }


        .film-card-body {
            padding: 20px;
            display: flex;
            flex-direction: column;
            flex-grow: 1; /* Biarkan body mengisi sisa ruang */
        }
        .film-card-title {
            font-size: 1.6em;
            font-weight: 700;
            color: #1a2a3a;
            margin-top: 0;
            margin-bottom: 10px;
            line-height: 1.2;
        }
        .film-release-date {
            font-size: 0.9em;
            color: #6a737d;
            margin-bottom: 15px;
        }
        .film-sinopsis-short {
            font-size: 0.95em;
            color: #555;
            margin-bottom: 20px;
            flex-grow: 1; /* Biarkan sinopsis mengambil ruang jika lebih panjang */
        }

        .film-card-actions {
            display: flex;
            flex-wrap: wrap; /* Wrap buttons on small screens */
            gap: 10px;
            margin-top: auto; /* Dorong ke bawah */
            justify-content: center; /* Pusatkan tombol jika hanya satu baris */
        }
        /* Tombol Aksi - Warna Minimalis */
        .action-btn {
            padding: 10px 15px;
            border-radius: 20px;
            text-decoration: none;
            font-weight: 600;
            font-size: 0.9em;
            transition: all 0.2s ease;
            text-align: center;
            flex-grow: 1;
            min-width: 120px;
            white-space: nowrap;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            border: 1px solid transparent; /* Default border */
        }
        .action-btn-primary { /* Lihat Jadwal & Pesan */
            background-color: #007bff; /* Primary blue for main action */
            color: white;
        }
        .action-btn-primary:hover {
            background-color: #0056b3;
            color: white;
            transform: translateY(-1px);
        }
        .action-btn-edit { /* EDIT BUTTON */
            background-color: #17a2b8; /* Info blue */
            color: white; /* Warna teks default */
            border-color: #17a2b8; /* Match background */
        }
        .action-btn-edit:hover {
            background-color: #138496; /* Darker info blue on hover */
            border-color: #138496;
            color: #e0e0e0; /* Warna teks saat di-hover, contoh: abu-abu terang */
            transform: translateY(-1px);
        }
        .action-btn-danger { /* Hapus */
            background-color: #dc3545; /* Red for destructive action */
            color: white;
            border: none;
            cursor: pointer;
        }
        .action-btn-danger:hover {
            background-color: #c82333;
            transform: translateY(-1px);
        }

        .delete-form {
            display: inline-block;
            width: auto;
            margin: 0;
            padding: 0;
        }


        /* Empty State (Consistent with other views) */
        .empty-state {
            text-align: center;
            padding: 50px 20px;
            background-color: #fefefe;
            border: 1px dashed #e0e0e0;
            border-radius: 10px;
            margin-top: 30px;
            color: #888;
            animation: popIn 0.5s ease-out;
        }
        @keyframes popIn {
            from { opacity: 0; transform: scale(0.9); }
            to { opacity: 1; transform: scale(1); }
        }
        .empty-state svg {
            margin-bottom: 20px;
            color: #ccc;
        }
        .empty-state p {
            font-size: 1.1em;
            margin-bottom: 25px;
        }
        .action-button { /* Re-use style for primary action button */
            background-color: #007bff;
            color: white;
            padding: 12px 25px;
            border-radius: 30px;
            text-decoration: none;
            font-weight: 600;
            transition: background-color 0.3s ease, transform 0.2s ease;
            box-shadow: 0 3px 8px rgba(0, 123, 255, 0.2);
        }
        .action-button:hover {
            background-color: #0056b3;
            transform: translateY(-2px);
        }

        /* Responsive Adjustments */
        @media (max-width: 992px) { /* Adjust grid columns for medium screens */
            .film-cards-grid {
                grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
                gap: 25px;
            }
            .film-card-header {
                height: 330px; /* Adjust height for smaller width */
            }
            .film-card-title {
                font-size: 1.4em;
            }
        }
        @media (max-width: 768px) {
            .film-list-container {
                margin: 20px 15px;
                padding: 20px;
            }
            .page-title {
                font-size: 2.2em;
            }
            .page-description {
                font-size: 1em;
            }
            .action-bar {
                text-align: center;
            }
            .add-film-button {
                width: 100%;
                box-sizing: border-box;
                justify-content: center;
            }
            .film-cards-grid {
                grid-template-columns: 1fr; /* Stack cards on very small screens */
                gap: 20px;
            }
            .film-card-header {
                height: 450px; /* Larger height when stacked */
            }
            .film-card-actions {
                flex-direction: column;
                align-items: stretch;
            }
            .action-btn {
                min-width: unset;
                width: 100%;
            }
        }
        @media (max-width: 480px) {
            .page-title {
                font-size: 2em;
            }
            .film-card-header {
                height: 350px; /* Adjust for smaller phone screens */
            }
            .film-card-title {
                font-size: 1.2em;
            }
            .film-card-body {
                padding: 15px;
            }
            .film-sinopsis-short {
                font-size: 0.9em;
            }
        }
    </style>