<?php
// views/pemesanan.php
// View ini digunakan untuk menampilkan daftar pemesanan pengguna.
// Variabel yang diharapkan: $pemesanan_list (array asosiatif daftar pemesanan pengguna)
// Variabel ini disediakan oleh PemesananController->list()

// Pesan sukses atau error dari sesi
if (isset($_SESSION['error_message'])) {
    echo "<p class='message error'>" . htmlspecialchars($_SESSION['error_message']) . "</p>";
    unset($_SESSION['error_message']);
}
if (isset($_SESSION['success_message'])) {
    echo "<p class='message success'>" . htmlspecialchars($_SESSION['success_message']) . "</p>";
    unset($_SESSION['success_message']);
}
?>

<div class="pemesanan-list-container">
    <h1 class="page-title">Riwayat Pemesanan Anda</h1>
    <p class="page-description">Temukan semua tiket bioskop yang telah Anda pesan di sini.</p>

    <?php if (!empty($pemesanan_list)): ?>
        <div class="table-responsive">
            <table class="pemesanan-table">
                <thead>
                    <tr>
                        <th>No.</th>
                        <th>Film</th>
                        <th>Studio</th>
                        <th>Tanggal Tayang</th>
                        <th>Waktu Tayang</th>
                        <th>Kursi</th>
                        <th>Harga</th>
                        <th>Tgl Pemesanan</th>
                        <th>Status</th>
                        <th class="text-center">Detail</th> </tr>
                </thead>
                <tbody>
                    <?php $no = 1; ?>
                    <?php foreach ($pemesanan_list as $pemesanan): ?>
                        <tr class="pemesanan-item-row">
                            <td data-label="No."><?php echo $no++; ?></td>
                            <td data-label="Film"><?php echo htmlspecialchars($pemesanan['film_judul']); ?></td>
                            <td data-label="Studio"><?php echo htmlspecialchars($pemesanan['studio_nama']); ?></td>
                            <td data-label="Tanggal Tayang"><?php echo htmlspecialchars($pemesanan['tanggal_tayang']); ?></td>
                            <td data-label="Waktu Tayang"><?php echo htmlspecialchars(substr($pemesanan['waktu_tayang'], 0, 5)); ?></td>
                            <td data-label="Kursi"><?php echo htmlspecialchars($pemesanan['kursi_nomor']); ?></td>
                            <td data-label="Harga" class="text-right">Rp <?php echo number_format($pemesanan['harga_saat_pemesanan'], 0, ',', '.'); ?></td>
                            <td data-label="Tgl Pemesanan"><?php echo htmlspecialchars($pemesanan['tanggal_pemesanan']); ?></td>
                            <td data-label="Status">
                                <span class="status-badge status-<?php echo htmlspecialchars($pemesanan['status_pemesanan']); ?>">
                                    <?php echo htmlspecialchars(ucfirst($pemesanan['status_pemesanan'])); ?>
                                </span>
                            </td>
                            <td data-label="Detail" class="text-center">
                                <button class="detail-button view-detail-btn"
                                        data-bs-toggle="modal" data-bs-target="#orderDetailModal"
                                        data-order-id="<?= htmlspecialchars($pemesanan['id']) ?>"
                                        title="Lihat Detail Pesanan">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-eye">
                                        <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                                        <circle cx="12" cy="12" r="3"></circle>
                                    </svg>
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php else: ?>
        <div class="empty-state">
            <svg xmlns="http://www.w3.org/2000/svg" width="60" height="60" viewBox="0 0 24 24" fill="none" stroke="#ccc" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-ticket">
                <path d="M18 8H6a2 2 0 0 0-2 2v4a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2v-4a2 2 0 0 0-2-2z"></path>
                <circle cx="6" cy="12" r="2"></circle>
                <circle cx="18" cy="12" r="2"></circle>
            </svg>
            <p>Anda belum memiliki riwayat pemesanan.</p>
            <a href="index.php?hal=film" class="action-button">Pesan Tiket Sekarang!</a>
        </div>
    <?php endif; ?>
</div>

<div class="modal fade" id="orderDetailModal" tabindex="-1" aria-labelledby="orderDetailModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered modal-lg"> <div class="modal-content ticket-modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="orderDetailModalLabel">Detail Tiket Pesanan #<span id="modalOrderId"></span></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body ticket-modal-body">
                <div id="modalOrderDetailsContent" class="ticket-details-wrapper">
                    <div class="loading-spinner"></div>
                </div>
            </div>
            <div class="modal-footer justify-content-between">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
            </div>
        </div>
    </div>
</div>

<style>
    /* Global Styles for body (from previous files, ensure consistency) */
    body {
        font-family: 'Poppins', sans-serif;
        background-color: #f0f2f5;
        color: #333;
        line-height: 1.6;
        margin: 0;
        padding: 0;
    }

    /* Message Styling (Success/Error) - Consistent across views */
    .message {
        padding: 12px 20px;
        margin: 20px auto;
        border-radius: 8px;
        text-align: center;
        font-weight: 600;
        max-width: 900px;
        box-sizing: border-box;
        animation: fadeInDown 0.5s ease-out;
        box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }

    @keyframes fadeInDown {
        from { opacity: 0; transform: translateY(-20px); }
        to { opacity: 1; transform: translateY(0); }
    }

    .message.error {
        background-color: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .message.success {
        background-color: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    /* Main Container for the Page */
    .pemesanan-list-container {
        max-width: 1000px; /* Slightly wider for tables */
        margin: 30px auto;
        padding: 30px;
        background-color: #ffffff;
        border-radius: 12px;
        box-shadow: 0 6px 20px rgba(0, 0, 0, 0.1);
        animation: fadeIn 0.8s ease-out;
    }

    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(20px); }
        to { opacity: 1; transform: translateY(0); }
    }

    .page-title {
        text-align: center;
        color: #1a2a3a;
        font-size: 2.5em;
        font-weight: 700;
        margin-bottom: 10px;
        position: relative;
        padding-bottom: 15px;
    }
    .page-title::after {
        content: '';
        width: 70px;
        height: 4px;
        background-color: #e96b56;
        position: absolute;
        bottom: 0;
        left: 50%;
        transform: translateX(-50%);
        border-radius: 2px;
    }

    .page-description {
        text-align: center;
        color: #6a737d;
        font-size: 1.1em;
        margin-bottom: 30px;
        line-height: 1.5;
    }

    /* Table Styling */
    .table-responsive {
        overflow-x: auto; /* Enable horizontal scroll on small screens */
        -webkit-overflow-scrolling: touch; /* Smooth scrolling for iOS */
        border-radius: 10px; /* Rounded corners for the responsive container */
        box-shadow: 0 2px 10px rgba(0,0,0,0.05); /* Subtle shadow for the table */
        border: 1px solid #e0e0e0;
    }

    .pemesanan-table {
        width: 100%;
        border-collapse: collapse;
        min-width: 768px; /* Minimum width for the table to prevent squishing on small screens */
    }

    .pemesanan-table thead tr {
        background-color: #f8f9fa; /* Light grey header */
        border-bottom: 2px solid #e9ecef;
    }

    .pemesanan-table th, .pemesanan-table td {
        padding: 15px;
        text-align: left;
        border: 1px solid #e9ecef; /* Lighter border for cells */
        font-size: 0.95em;
        vertical-align: middle;
    }

    .pemesanan-table th {
        color: #495057;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .pemesanan-table tbody tr:nth-child(even) {
        background-color: #fcfcfc; /* Zebra striping */
    }

    .pemesanan-table tbody tr:hover {
        background-color: #f2f2f2; /* Hover effect */
        transition: background-color 0.2s ease;
    }

    .pemesanan-table td {
        color: #343a40;
    }

    .pemesanan-table .text-right {
        text-align: right;
    }
    .pemesanan-table .text-center {
        text-align: center;
    }

    /* Status Badges */
    .status-badge {
        display: inline-block;
        padding: 6px 12px;
        border-radius: 20px; /* Pill shape */
        font-size: 0.85em;
        font-weight: 600;
        text-transform: capitalize;
        white-space: nowrap; /* Prevent text wrapping */
    }

    .status-pending {
        background-color: #fff3cd;
        color: #856404;
        border: 1px solid #ffc107;
    }
    .status-paid {
        background-color: #d4edda;
        color: #155724;
        border: 1px solid #28a745;
    }
    .status-cancelled {
        background-color: #f8d7da;
        color: #721c24;
        border: 1px solid #dc3545;
    }

    /* Detail Button */
    .detail-button {
        background-color: #e96b56; /* Info */
        color: white;
        border: none;
        border-radius: 5px;
        padding: 8px 12px;
        cursor: pointer;
        transition: background-color 0.3s ease, transform 0.2s ease;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        gap: 5px;
    }
    .detail-button:hover {
        background-color:rgb(231, 103, 81);
        transform: translateY(-1px);
    }
    .detail-button svg {
        vertical-align: middle;
    }

    /* Empty State */
    .empty-state {
        text-align: center;
        padding: 50px 20px;
        background-color: #fefefe;
        border: 1px dashed #e0e0e0;
        border-radius: 10px;
        margin-top: 30px;
        color: #888;
        animation: popIn 0.5s ease-out;
    }

    @keyframes popIn {
        from { opacity: 0; transform: scale(0.9); }
        to { opacity: 1; transform: scale(1); }
    }

    .empty-state svg {
        margin-bottom: 20px;
        color: #ccc;
    }

    .empty-state p {
        font-size: 1.1em;
        margin-bottom: 25px;
    }

    .action-button {
        background-color: #007bff;
        color: white;
        padding: 12px 25px;
        border-radius: 30px;
        text-decoration: none;
        font-weight: 600;
        transition: background-color 0.3s ease, transform 0.2s ease;
        box-shadow: 0 3px 8px rgba(0, 123, 255, 0.2);
    }
    .action-button:hover {
        background-color: #0056b3;
        transform: translateY(-2px);
    }

    /* --- MODAL STYLING (Dipercantik & 2 Kolom Grid) --- */
    .modal-dialog.modal-lg { /* UKURAN MODAL BARU: modal-lg */
        max-width: 800px; /* Lebar maksimal yang lebih besar */
    }

    .ticket-modal-content {
        border-radius: 15px;
        box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        overflow: hidden;
        background: linear-gradient(145deg, #f8f9fa, #e9ecef);
    }

    .modal-header {
        background-color: #343a40;
        color: white;
        border-bottom: none;
        padding: 20px 30px;
    }
    .modal-header .modal-title {
        color: white;
        font-weight: 700;
        font-size: 1.3em;
        letter-spacing: 0.5px;
    }
    .modal-header .btn-close {
        filter: invert(1);
        opacity: 0.8;
    }

    .ticket-modal-body {
        padding: 0;
    }

    .ticket-details-wrapper {
        padding: 30px;
        background-color: #ffffff;
        position: relative;
        border-radius: 0 0 15px 15px;
    }
    /* Efek "sobekan" tiket */
    .ticket-details-wrapper::before,
    .ticket-details-wrapper::after {
        content: '';
        position: absolute;
        width: 30px;
        height: 30px;
        background-color: #f0f2f5;
        border-radius: 50%;
        top: -15px;
    }
    .ticket-details-wrapper::before {
        left: -15px;
    }
    .ticket-details-wrapper::after {
        right: -15px;
    }

    /* Loading Spinner */
    .loading-spinner {
        border: 4px solid #f3f3f3;
        border-top: 4px solid #007bff;
        border-radius: 50%;
        width: 40px;
        height: 40px;
        animation: spin 1s linear infinite;
        margin: 50px auto;
        display: block;
    }
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }

    /* Ticket Details Content */
    .ticket-section {
        margin-bottom: 25px;
        text-align: center;
    }
    .ticket-section.header-info {
        margin-bottom: 30px;
        padding-bottom: 15px;
        border-bottom: 1px dashed #eee;
    }
    .ticket-movie-title {
        font-size: 1.8em;
        font-weight: 700;
        color: #dc3545;
        margin-bottom: 5px;
        line-height: 1.2;
    }
    .ticket-subtitle {
        font-size: 0.9em;
        color: #888;
        margin-bottom: 3px;
    }
    .ticket-user {
        font-size: 0.95em;
        color: #555;
        font-weight: 500;
    }

    /* --- GRID LAYOUT UNTUK DETAIL UTAMA --- */
    .ticket-main-grid {
        display: grid;
        grid-template-columns: 1fr 1fr; /* Dua kolom dengan lebar yang sama */
        gap: 20px; /* Jarak antar kolom */
        margin-bottom: 25px;
        text-align: left; /* Teks di dalam grid rata kiri */
    }

    .grid-column-left, .grid-column-right {
        padding-right: 10px; /* Sedikit padding internal */
        border-right: 1px solid #f0f0f0; /* Pemisah kolom */
    }
    .grid-column-right {
        padding-left: 10px;
        border-right: none; /* Hapus border kanan kolom terakhir */
    }
    /* Untuk Mobile, kembali ke satu kolom */
    @media (max-width: 576px) {
        .ticket-main-grid {
            grid-template-columns: 1fr; /* Satu kolom di mobile */
            gap: 0;
        }
        .grid-column-left, .grid-column-right {
            border-right: none;
            padding-right: 0;
            padding-left: 0;
        }
        .grid-column-left {
            padding-bottom: 15px; /* Jarak antar bagian di mobile */
            border-bottom: 1px dashed #eee;
            margin-bottom: 15px;
        }
    }


    /* Detail Info Group (dalam grid) */
    .ticket-info-group {
        display: flex;
        align-items: center;
        justify-content: space-between; /* Meratakan item ke kiri dan kanan */
        margin-bottom: 12px; /* Sedikit lebih rapat */
        padding: 5px 0; /* Padding lebih kecil */
        border-bottom: 1px dashed #eee; /* Garis putus-putus */
    }
    .ticket-info-group:last-of-type {
        border-bottom: none;
        margin-bottom: 0;
    }

    .ticket-icon {
        /* Ukuran dan warna icon */
        font-size: 1.3em; /* Ukuran jika pakai emoji */
        color: #e96b56;
        margin-right: 12px;
        flex-shrink: 0;
        width: 25px;
        text-align: center;
    }
    .ticket-icon svg { /* Styling SVG Feather Icons */
        width: 20px; /* Ukuran SVG */
        height: 20px;
        vertical-align: middle;
        stroke: #e96b56; /* Warna ikon SVG */
        stroke-width: 2;
    }


    .ticket-label {
        font-weight: 600;
        color: #555;
        flex-grow: 1;
        text-align: left;
    }
    .ticket-value {
        font-weight: 500;
        color: #333;
        flex-shrink: 0;
        text-align: right;
        margin-left: 10px;
    }
    .ticket-value.highlight {
        color: #e96b56;
        font-weight: 700;
        font-size: 1em;
    }
    .ticket-value.price {
        color: #e96b56;
        font-size: 1.1em;
        font-weight: 700;
    }
    .ticket-value.status {
        font-weight: 700;
        text-transform: capitalize;
    }
    .ticket-value.status.paid { color: #28a745; }
    .ticket-value.status.pending { color: #ffc107; }
    .ticket-value.status.cancelled { color: #dc3545; }

    /* QR Code Placeholder */
    .qr-code-placeholder {
        text-align: center;
        margin: 30px auto 15px auto;
        padding: 10px;
        background-color: #f0f0f0;
        border-radius: 10px;
        max-width: 180px;
        border: 1px dashed #ccc;
    }
    .qr-code-placeholder img {
        width: 120px;
        height: 120px;
        border: 1px solid #ccc;
        background-color: white;
        display: block;
        margin: 0 auto 8px auto;
    }
    .qr-code-placeholder p {
        font-size: 0.85em;
        color: #777;
        margin: 0;
    }

    /* Fine Print / Additional Notes */
    .fine-print {
        font-size: 0.8em;
        color: #888;
        text-align: center;
        margin-top: 15px;
        padding-top: 10px;
        border-top: 1px dashed #eee;
    }

    /* Print Button in Modal Footer */
    .modal-footer .print-ticket-btn {
        background-color: #6c757d;
        border-color: #6c757d;
        color: white;
        transition: background-color 0.3s ease;
    }
    .modal-footer .print-ticket-btn:hover {
        background-color: #5a6268;
        border-color: #545b62;
    }
    .modal-footer .print-ticket-btn svg {
        margin-right: 5px;
    }

    /* Responsive Adjustments for Modal */
    @media (max-width: 576px) {
        .modal-dialog.modal-lg { /* Adjust modal-lg for mobile */
            margin: 0.5rem;
            max-width: unset;
            width: auto;
        }
        .ticket-modal-content {
            border-radius: 10px;
        }
        .ticket-details-wrapper {
            padding: 20px;
        }
        .ticket-details-wrapper::before,
        .ticket-details-wrapper::after {
            width: 20px;
            height: 20px;
            top: -10px;
            left: -10px;
            right: -10px;
        }
        .modal-header, .modal-footer {
            padding: 15px 20px;
        }
        .modal-header .modal-title {
            font-size: 1.1em;
        }
        /* Mobile: Grid kembali ke 1 kolom, detail info group tetap horizontal di dalamnya */
        .ticket-main-grid {
            grid-template-columns: 1fr;
            gap: 0;
            margin-bottom: 15px;
        }
        .grid-column-left, .grid-column-right {
            border-right: none;
            padding-right: 0;
            padding-left: 0;
        }
        .grid-column-left {
            padding-bottom: 15px;
            border-bottom: 1px dashed #eee;
            margin-bottom: 15px;
        }

        .ticket-info-group { /* Tetap horizontal di mobile */
            flex-direction: row;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 8px;
            padding: 3px 0;
        }
        .ticket-icon {
            font-size: 1.1em;
            margin-right: 8px;
            width: 20px;
        }
        .ticket-label {
            min-width: unset;
            margin-bottom: 0;
        }
        .ticket-value {
            text-align: right;
            font-size: 0.85em;
            margin-left: 5px;
        }
        .ticket-value.price, .ticket-value.highlight {
            font-size: 0.95em;
        }
        .qr-code-placeholder {
            margin: 15px auto 10px auto;
            max-width: 120px;
        }
        .qr-code-placeholder img {
            width: 90px;
            height: 90px;
        }
        .fine-print {
            font-size: 0.75em;
        }
        .print-ticket-btn {
            font-size: 0.9em;
            padding: 8px 15px;
        }
    }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const orderDetailModal = document.getElementById('orderDetailModal');
    const modalOrderId = document.getElementById('modalOrderId');
    const modalOrderDetailsContent = document.getElementById('modalOrderDetailsContent');

    if (orderDetailModal) {
        orderDetailModal.addEventListener('show.bs.modal', function (event) {
            const button = event.relatedTarget;
            const orderId = button.getAttribute('data-order-id');

            modalOrderId.textContent = orderId;
            modalOrderDetailsContent.innerHTML = '<div class="loading-spinner"></div>';

            fetch(`index.php?hal=pemesanan_get_detail&id=${orderId}`)
                .then(response => {
                    if (!response.ok) {
                        if (response.status === 403) {
                            throw new Error('Akses ditolak. Ini bukan pesanan Anda.');
                        }
                        if (response.status === 404) {
                            throw new Error('Detail pesanan tidak ditemukan.');
                        }
                        throw new Error('Network response was not ok');
                    }
                    return response.json();
                })
                .then(data => {
                    if (data && !data.error) {
                        let statusColorClass = '';
                        if (data.status_pemesanan === 'paid') {
                            statusColorClass = 'paid';
                        } else if (data.status_pemesanan === 'pending') {
                            statusColorClass = 'pending';
                        } else if (data.status_pemesanan === 'cancelled') {
                            statusColorClass = 'cancelled';
                        }

                        let paymentMethodDisplay = data.metode_pembayaran ? htmlspecialchars(data.metode_pembayaran.replace(/_/g, ' ').replace(/\b\w/g, c => c.toUpperCase())) : 'Tidak Diketahui';

                        const userName = data.user_fullname ? htmlspecialchars(data.user_fullname) : 'Pengguna Tidak Diketahui';
                        const userUsername = data.user_username ? `(@${htmlspecialchars(data.user_username)})` : '';


                        modalOrderDetailsContent.innerHTML = `
                            <div class="ticket-section header-info">
                                <p class="ticket-movie-title">${htmlspecialchars(data.film_judul)}</p>
                                <p class="ticket-subtitle">Booking ID: ${htmlspecialchars(data.id)}</p>
                                <p class="ticket-user">Oleh: ${userName} ${userUsername}</p>
                            </div>

                            <div class="ticket-main-grid">
                                <div class="grid-column-left">
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-film"><rect x="2" y="2" width="20" height="20" rx="2.18" ry="2.18"></rect><line x1="7" y1="2" x2="7" y2="22"></line><line x1="17" y1="2" x2="17" y2="22"></line><line x1="2" y1="12" x2="22" y2="12"></line><line x1="2" y1="7" x2="7" y2="7"></line><line x1="2" y1="17" x2="7" y2="17"></line><line x1="17" y1="7" x2="22" y2="7"></line><line x1="17" y1="17" x2="22" y2="17"></line></svg></span>
                                        <span class="ticket-label">Film</span>
                                        <span class="ticket-value highlight">${htmlspecialchars(data.film_judul)}</span>
                                    </div>
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-monitor"><rect x="2" y="3" width="20" height="14" rx="2" ry="2"></rect><line x1="8" y1="21" x2="16" y2="21"></line><line x1="12" y1="17" x2="12" y2="21"></line></svg></span>
                                        <span class="ticket-label">Studio</span>
                                        <span class="ticket-value highlight">${htmlspecialchars(data.studio_nama)}</span>
                                    </div>
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-calendar"><rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect><line x1="16" y1="2" x2="16" y2="6"></line><line x1="8" y1="2" x2="8" y2="6"></line><line x1="3" y1="10" x2="21" y2="10"></line></svg></span>
                                        <span class="ticket-label">Tanggal Tayang</span>
                                        <span class="ticket-value">${htmlspecialchars(data.tanggal_tayang)}</span>
                                    </div>
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-clock"><circle cx="12" cy="12" r="10"></circle><polyline points="12 6 12 12 16 14"></polyline></svg></span>
                                        <span class="ticket-label">Waktu Tayang</span>
                                        <span class="ticket-value">${htmlspecialchars(data.waktu_tayang).substring(0, 5)} WIB</span>
                                    </div>
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-chair"><path d="M5 14s1-1 4-1 4 1 4 1V3H5z"></path><path d="M16 14s1-1 4-1 4 1 4 1V3h-9z"></path></svg></span>
                                        <span class="ticket-label">Nomor Kursi</span>
                                        <span class="ticket-value highlight">${htmlspecialchars(data.kursi_nomor)}</span>
                                    </div>
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-dollar-sign"><line x1="12" y1="1" x2="12" y2="23"></line><path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"></path></svg></span>
                                        <span class="ticket-label">Harga per Kursi</span>
                                        <span class="ticket-value price">Rp ${new Intl.NumberFormat('id-ID', { minimumFractionDigits: 0, maximumFractionDigits: 0 }).format(data.harga_saat_pemesanan)}</span>
                                    </div>
                                </div>
                                <div class="grid-column-right">
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-calendar-check"><rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect><line x1="16" y1="2" x2="16" y2="6"></line><line x1="8" y1="2" x2="8" y2="6"></line><line x1="3" y1="10" x2="21" y2="10"></line><polyline points="9 14 11 16 15 12"></polyline></svg></span>
                                        <span class="ticket-label">Tgl Pemesanan</span>
                                        <span class="ticket-value">${htmlspecialchars(data.tanggal_pemesanan)}</span>
                                    </div>
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-credit-card"><rect x="1" y="4" width="22" height="16" rx="2" ry="2"></rect><line x1="1" y1="10" x2="23" y2="10"></line></svg></span>
                                        <span class="ticket-label">Metode Bayar</span>
                                        <span class="ticket-value">${paymentMethodDisplay}</span>
                                    </div>
                                    <div class="ticket-info-group">
                                        <span class="ticket-icon"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-check-circle"><path d="M22 11.08V12a10 10 0 1 1-5.93-8.15"></path><polyline points="22 4 12 14.01 9 11.01"></polyline></svg></span>
                                        <span class="ticket-label">Status</span>
                                        <span class="ticket-value status ${statusColorClass}">${htmlspecialchars(data.status_pemesanan.charAt(0).toUpperCase() + data.status_pemesanan.slice(1))}</span>
                                    </div>

                                    <div class="qr-code-placeholder">
                                        <img src="/bioskop/assets/img/qr.png" alt="QR Code Tiket"> <p>Scan untuk Verifikasi</p>
                                    </div>
                                </div>
                            </div>

                            <p class="fine-print">Harap tunjukkan QR Code ini kepada petugas untuk verifikasi tiket Anda. Tiket tidak dapat diuangkan kembali.</p>
                        `;
                    } else {
                        modalOrderDetailsContent.innerHTML = '<p class="error-message">Detail pesanan tidak ditemukan atau terjadi kesalahan.</p>';
                    }
                })
                .catch(error => {
                    console.error('Error fetching order details:', error);
                    modalOrderDetailsContent.innerHTML = `<p class="error-message">Gagal memuat detail pesanan. ${error.message || 'Terjadi kesalahan tidak dikenal.'}</p>`;
                });
        });
    }

    // Helper function for HTML escaping in JavaScript
    function htmlspecialchars(str) {
        if (typeof str !== 'string' && typeof str !== 'number') return str;
        str = String(str);
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return str.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
});
</script>